`timescale 1s/100ms

module testbench();

    wire clock;
    reg reset_;

    clock_generator clk(
        .clock(clock)
    );

    wire soc_x; reg eoc_x;
    reg [7:0] x;
    wire soc_y; reg eoc_y;
    reg [7:0] y;
    wire out;

    ABC dut (
        .soc_x(soc_x), .eoc_x(eoc_x), .x(x),
        .soc_y(soc_y), .eoc_y(eoc_y), .y(y),
        .out(out),
        .reset_(reset_), .clock(clock)
    );

    // simulation variables, representing the actual temperatures
    // x may give a different temperature, y never lies
    reg [7:0] actual_temperature;
    reg [7:0] x_temperature;

    // simulation variables, used to measure time distances
    realtime prev_sample;
    realtime current_sample;
    realtime diff;

    // Debug variables to highlight errors in waveform diagram
    reg error_value;
    initial error_value = 0;
    always @(posedge error_value) #1
        error_value = 0;

    reg error_period;
    initial error_period = 0;
    always @(posedge error_period) #1
        error_period = 0;
    
    initial
        begin
            $dumpfile("waveform.vcd");
            $dumpvars;

            //the following structure is used to wait for expected signals, and fail if too much time passes
            fork : f
                begin
                    #100000;
                    $display("Timeout - waiting for signal failed");
                    disable f;
                end
            //actual tests start here
                begin
                    prev_sample = 0;
                    eoc_x = 1; x = 8'd40;
                    eoc_y = 1; y = 8'd40;

                    //reset phase
                    reset_ = 0; #(clk.HALF_PERIOD + 1);
                    
                    //just after first posedge
                    reset_ = 1;

                    if(out !== 0)
                        $display("out is not 0 after reset");

                    if(soc_x !== 0)
                        $display("soc_x is not 0 after reset");

                    if(soc_y !== 0)
                        $display("soc_y is not 0 after reset");

                    fork
                        begin : Sensore_X
        	                reg [3:0] i;
                                                        
                            for(i = 0; i < 9; i++) begin
                                @(posedge soc_x);
                                current_sample = $realtime;
                                if(prev_sample != 0) begin
                                    diff = (current_sample - prev_sample)/(2*clk.HALF_PERIOD);
                                    if( diff != 60 ) begin
                                        $display("Wrong timing: expected 60 clocks, got %g instead", diff);
                                        error_period = 1;
                                    end
                                end
                                prev_sample = current_sample;

                                #(1*clk.HALF_PERIOD);
                                {actual_temperature, x_temperature} = get_testcase(i[2:0]);
                                #(2*clk.HALF_PERIOD);
                                x = 8'hXX;
                                eoc_x = 0;
                                #(0.1);
                                @(negedge soc_x)
                                #(3*2*clk.HALF_PERIOD);
                                x = x_temperature;
                                #(0.1);
                                eoc_x = 1;
                                #(0.1);
                            end

                            #(60*2*clk.HALF_PERIOD);
                            disable Sensore_Y;
                            disable out_check;
                        end

                        begin : Sensore_Y
        	                
                            while(1) begin
                                @(posedge soc_y);
                                #(5*2*clk.HALF_PERIOD);
                                y = 8'hXX;
                                eoc_y = 0;
                                #(0.1);
                                @(negedge soc_y)
                                #(5*2*clk.HALF_PERIOD);
                                y = actual_temperature;
                                #(0.1);
                                eoc_y = 1;
                                #(0.1);
                            end
                        end

                        begin : out_check
        	                while(1) begin
                                @(posedge out);
                                if(x_temperature <= 8'd180 || x_temperature !== actual_temperature) begin
                                    $display("Unexpected out signal: x is %d and y is %d", x_temperature, actual_temperature);
                                    error_value = 1;
                                end
                            end
                        end
                        
                    join                   

                    disable f;
                end   
            join

            $finish;
        end


    function automatic [18:0] get_testcase;
        input [2:0] i;

        reg [7:0] actual_temperature, x_temperature;
        
        begin
            casex(i)
                3'b000: begin
                    actual_temperature = 8'd80; // 60.0 C
                    x_temperature = 8'd80; // 60.0 C
                end
                3'b001: begin
                    actual_temperature = 8'd120; // 60.0 C
                    x_temperature = 8'd120; // 60.0 C
                end
                3'b010: begin
                    actual_temperature = 8'd190; // 95.0 C
                    x_temperature = 8'd190; // 95.0 C
                end
                3'b011: begin
                    actual_temperature = 8'd200; // 100.0 C
                    x_temperature = 8'd190; // 95.0 C
                end
                3'b100: begin
                    actual_temperature = 8'd100; // 50.0 C
                    x_temperature = 8'd200; // 100.0 C
                end
                3'b101: begin
                    actual_temperature = 8'd49; // 24.5 C
                    x_temperature = 8'd49; // 24.5 C
                end
                3'b110: begin
                    actual_temperature = 8'd120; // 60.0 C
                    x_temperature = 8'd200; // 100.0 C
                end
                3'b111: begin
                    actual_temperature = 8'd127; // 63.5 C
                    x_temperature = 8'd127; // 63.5 C
                end
            endcase

            get_testcase = {actual_temperature, x_temperature};
        end
    endfunction
endmodule

// generatore del segnale di clock
module clock_generator(
    clock
);
    output clock;

    parameter HALF_PERIOD = 5;

    reg CLOCK;
    assign clock = CLOCK;

    initial CLOCK <= 0;
    always #HALF_PERIOD CLOCK <= ~CLOCK;

endmodule