module testbench();
    wire clock;
    clock_generator clk(
        .clock(clock)
    );
    reg reset_;

    wire [15:0] addr;
    wire [7:0] data;
    wire mr_, mw_;

    reg soc;
    wire eoc;
    reg [15:0] p;
    wire [15:0] max;

    ABC dut (
        .addr(addr), .data(data), .mr_(mr_), .mw_(mw_),
        .soc(soc), .eoc(eoc), .p(p), .max(max),
        .clock(clock), .reset_(reset_) 
    );

    ram_16Kx8 ram (
        .addr(addr), .data(data),
        .s_(1'b0), .mr_(mr_), .mw_(mw_)
    );

    // linee di errore e sampling, utili per il debug
    reg error, sampling;
    initial error = 0;
    always @(posedge error) #1 
        error = 0;
    initial sampling = 0;
    always @(posedge sampling) #1 
        sampling = 0;

    initial begin
        $dumpfile("waveform.vcd");
        $dumpvars;

        //the following structure is used to wait for expected signals, and fail if too much time passes
        fork : f
            begin
                #100000;
                $display("Timeout - waiting for signal failed");
                disable f;
            end
        //actual tests start here
            begin: test
                reg [6:0] i;
                reg [15:0] p_t, max_t;

                soc = 0;

                //reset phase
                reset_ = 0;
                #(clk.HALF_PERIOD);
                @(posedge clock);
                #(clk.HALF_PERIOD);
                reset_ = 1;

                if(eoc !== 1) begin
                    error = 1;
                    $display("eoc should be 1 after reset");
                end

                for(i = 0; i < 64; i++) begin
                    {p_t, max_t} = get_testcase(i[4:0]);
                    p <= p_t;
                    #1;
                    soc <= 1;
                    @(negedge eoc);
                    p <= 16'hXXXX;
                    soc <= 0;
                    @(posedge eoc);
                    sampling = 1;
                    if(max !== max_t) begin
                        error = 1;
                        $display("%g-th result is %g , expected %g instead", i, max, max_t);
                    end
                end

                disable f;
            end
        join

        #10;
        $finish;
    end
    
    function automatic [36:0] get_testcase;
        input [4:0] i;
        
        reg [15:0] p;
        reg [15:0] max;

        begin
            casex(i[0])
                1'd0: begin
                    p = 16'h1234; max = 16'h040A;    // 00FF, 0102, 040A
                end
                1'd1: begin
                    p = 16'h1357; max = 16'h2468;    // 2468
                end
                default: begin
                    p = 16'h0000; max = 16'h0000;
                end
            endcase
            get_testcase = {p, max};
        end
    endfunction

endmodule

module ram_16Kx8 (
    addr, s_, mr_, mw_,
    data
);
    input [15:0] addr;
    input s_, mr_, mw_;
    
    inout [7:0] data;
    reg DIR;
    reg [7:0] OUTBYTE;
    assign data = DIR ? OUTBYTE : 8'bZZ;

    reg [7:0] mem [0:65535];

    initial begin
        // initialize memory as desired
        {mem[16'h1234], mem[16'h1235]} <= 16'h00FF;
        {mem[16'h1236], mem[16'h1237]} <= 16'hAAAA;
        {mem[16'hAAAA], mem[16'hAAAB]} <= 16'h0102;
        {mem[16'hAAAC], mem[16'hAAAD]} <= 16'hBBBB;
        {mem[16'hBBBB], mem[16'hBBBC]} <= 16'h040A;
        {mem[16'hBBBD], mem[16'hBBBE]} <= 16'h0000;

        {mem[16'h1357], mem[16'h1358]} <= 16'h2468;
        {mem[16'h1359], mem[16'h135A]} <= 16'h0000;
    end

    always @(addr or s_ or mw_) if (s_ == 0 && mw_ == 0) #1
    begin
        mem[addr] <= data;
        DIR <= 0;
    end

    always @(addr or s_ or mr_) if (s_ == 0 && mr_ == 0) #1
    begin
        OUTBYTE <= mem[addr];
        DIR <= 1;
    end

endmodule

// generatore del segnale di clock
module clock_generator(
    clock
);
    output clock;

    parameter HALF_PERIOD = 5;

    reg CLOCK;
    assign clock = CLOCK;

    initial CLOCK <= 0;
    always #HALF_PERIOD CLOCK <= ~CLOCK;

endmodule
