
/*
 * =====================================================================================
 *
 *       Filename:  SwiSpot.c
 *
 *    Description: SwiSpot is a command line utility for predicting
 * 							   riboswitch alternative configurations.
 *
 *        Version:  0.2.3
 *        Created:  22.02.2016 15:23:58
 *       Revision:  none
 *       Compiler:  clang
 *
 *         Author:  BARSACCHI MARCO (BM), BARSACCHIMARCO@GMAIL.COM
 *   Organization:
 *
 *  LICENSE:
 *  Copyright (C) 2015  Barsacchi Marco
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * =====================================================================================
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <ctype.h>
#include <sys/time.h>
// Regex
#include <regex.h>
// Start of getopt
//#include <unistd.h>
#include <getopt.h>
// End of getopt
// ViennaRNA Include
#include "fold.h"
#include "utils.h"
#include "fold_vars.h"
#include "part_func.h"
#include "RNAstruct.h"
#include "params.h"
// Clustering
#include "cluster.h"
// SwiSpot header
#include "SwiSpot.h"
#include "confparse.h"

/*
 * COMMAND LINE PARSING STRUCTURES
 *
 *
 */
struct globalArgs_t {
    double temp;     /* -t option temperature*/
    char* sequence;  /* -s option Input sequence */
    int numStru;     /* -n optionNumber of structures*/
    int wFile;       /* -o output to file*/
    int probability; /* -p option, probability*/
    char* filePath;  /* Path, option specified by -p */
    int logging;     /* -l option, verbosity turned on */
    char* confPath;
} globalArgs;

// Define how options are used
static const char* optString = "t:s:n:o:c:pvlh";

static const struct option longOpts[] = { { "temperature", required_argument, NULL, 't' },
    { "sequence", required_argument, NULL, 's' }, { "numstruct", required_argument, NULL, 'n' },
    { "output", required_argument, NULL, 'o' }, { "confpath", required_argument, NULL, 'c' },
    { "probability", no_argument, NULL, 'p' }, { "help", no_argument, NULL, 'h' },
    { "logging", no_argument, NULL, 'l' }, { "version", no_argument, NULL, 'v' }, { NULL, no_argument, NULL, 0 } };

/*
 * SOME REGEX CONSTANTS
 *
 *
 */
// Terminators
static const char* regexTerm1 = "y{3,}[ACGT]{3,8}x{3,}.T{3,}.{,20}$";
static const char* regexTerm2 = "x{8,}T{3,}.{,20}$";
// ShineDelgarno sequences
static const char* regexSD1 = "AGGAG.{1,50}$";
static const char* regexSD2 = "AGAAG.{1,50}$";
static const char* regexSD3 = "AAAGG.{1,50}$";
static const char* regexSD4 = "AGGAG.{1,50}$";
static const char* regexSD5 = "AGAAG.{1,50}$";
static const char* regexSD6 = "AAAGG.{1,50}$";
static const char* regexSD7 = "AGGAG.{1,50}$";
static const char* regexSD8 = "AGAAG.{1,50}$";
static const char* regexSD9 = "AAAGG.{1,50}$";
static const char* regexSD10 = "AGGAG.{1,50}$";
static const int sdLen = 5;
float sp = 1.0;
float sn = 0;

/*
 * Help and usage functions
 */
void print_usage()
{
    printf(
        "Usage: SwiSpot -s sequence [-t temperature] [-o output] [-c confpath] [-n number of structures] [-pvlh]\n\n");
}
void print_help()
{
    printf(HELPSTRING);
}

// MAIN
int main(int argc, const char* argv[])
{
    // Timeval structure, used to evaluate computation time.
    struct timeval tval_before, tval_after, tval_result;
    gettimeofday(&tval_before, NULL);
    // Some definitions
    float e1, kT, eUP, eDOWN;
    char *structure, *foldStructUP, *foldStructDOWN;
    int len;
    int nMin = 4, nMax = 10;
    free_pf_arrays();

    // TO UNCOMMENT IN RELEASE
	init_rand();

    // Initialization and parsing of command line arguments
    int opt = 0;
    globalArgs.temp = 37.0;
    globalArgs.sequence = NULL;
    globalArgs.numStru = 100; /* Default 100 structures*/
    globalArgs.wFile = 0;
    globalArgs.filePath = NULL;
    globalArgs.probability = 0;
    globalArgs.logging = 0;
    globalArgs.confPath = NULL;
    temperature = 37.0;

    /*
     * Command Line Parsing
     * Using GNU getopt library
     *
     *
     */
    int* longIndex;
    while((opt = getopt_long(argc, argv, optString, longOpts, &longIndex)) != -1) {
	switch(opt) {
	case 't':
	    globalArgs.temp = atof(optarg);
	    break;
	case 's':
	    globalArgs.sequence = optarg;
	    break;
	case 'n':
	    globalArgs.numStru = atoi(optarg);
	    break;
	case 'l':
	    globalArgs.logging = 1;
	    printf("Verbosity turned on. \n");
	    break;
	case 'c':
	    globalArgs.confPath = optarg;
	    break;
	case 'o':
	    globalArgs.wFile = 1;
	    globalArgs.filePath = optarg;
	    break;
	case 'p':
	    globalArgs.probability = 1;
	    break;
	case 'v':
	    printf("SwiSpot version %s\n", SWISPOTVERSION);
	    exit(EXIT_SUCCESS);
	case 'h': /* fall-through is intentional */
	    print_help();
	    exit(EXIT_SUCCESS);
	case '?':
	    print_usage();
	    exit(EXIT_FAILURE);
	case 0: /* long option without a short arg */
	    break;
	default:
	    break;
	}
    }
    // Enforce mandatory arguments
    if(!globalArgs.sequence) {
	print_usage();
	printf("\nSequence is mandatory!\n");
	exit(EXIT_FAILURE);
    }

    // printf("%s \n", globalArgs.sequence);

    /*
     * Parsing Configuration File
     */

    if(globalArgs.confPath != NULL) {
	struct confFile_t myConf;
	fileParse(&myConf, globalArgs.confPath);
	sp = myConf.sp;
	sn = myConf.sn;
	if(globalArgs.logging == 1) {
	    printf("VERBOSE:\n\tParameters read: sp %f,sn %f \n", myConf.sp, myConf.sn);
	}
    }

    if(globalArgs.logging == 1) {
	printf("VERBOSE:\n\tTemperature: %f \n", globalArgs.temp);
    }

    char* seq = (char*)malloc(strlen(globalArgs.sequence) + 1);
    strcpy(seq, globalArgs.sequence);
    len = strlen(seq);
    if(verifySequence(seq, len)) {
	exit(EXIT_FAILURE);
    }
    printf("Sequence length: %d \n", len);
    printf("%s\n", seq);
	
    // Some init
    //structure = (char*)space(sizeof(char) * (len));
    //foldStructUP = (char*)space(sizeof(char) * (len));
    //foldStructDOWN = (char*)space(sizeof(char) * (len));
	structure = (char*)malloc(strlen(globalArgs.sequence) + 1);
	foldStructUP = (char*)malloc(strlen(globalArgs.sequence) + 1);
	foldStructDOWN = (char*)malloc(strlen(globalArgs.sequence) + 1);
    e1 = fold(seq, structure);

    kT = (temperature + 273.15) * 1.98717 / 1000.; /* kT in kcal/mol */
    pf_scale = exp(-e1 / kT / strlen(seq));
    st_back = 1;
    e1 = pf_fold(seq, structure);
    if(globalArgs.logging == 1) {
	printf("VERBOSE:\n\t%s\n\t%f\n", structure, e1);
    }
    // Generating base-pair probability matrix, scaled at fixed temperature
    FLT_OR_DBL* partF = export_bppm();
	
	double **matrix = (double**)malloc(len * sizeof(double*));
	int i, j;
    for(i = 0; i < len; i++) {
		matrix[i] = (double *)malloc(len * sizeof(double));
		for (j=0; j < len; j++){
			matrix[i][j] = 0;
		}
	}
	
	
    for(i = 1; i < len; i++) {
		for(j = i + 1; j <= len; j++) {
	    // matrix = pMAtrix + pMatrix.T
	    matrix[i - 1][j - 1] = (double)partF[iindx[i] - j];
		
	    matrix[j - 1][i - 1] = (double)partF[iindx[i] - j];
		
	}
    }
	/* Printing probability matrix
	for(i = 1; i < len; i++) {
	for(j = i + 1; j <= len; j++) {
	    // matrix = pMAtrix + pMatrix.T
	    printf("%f,%f",matrix[i - 1][j - 1],matrix[j - 1][i - 1]);
	}
	printf("\n");
    }
	*/
	
    // Clustering, using cluster library
    int nclust = 2, npass = 10, eN = globalArgs.numStru;
    int clustid[eN];
    double** distMat;
    double error;
    int ifound = 0;
    // Generating eN structures
    /*
	char* structures[eN];
	for(j = 0; j < eN; j++) {
		structures[j] = pbacktrack(seq);
    }
	 */
	char **structures = (char**)malloc(eN* sizeof(char*));
	for (int j =0; j<eN; j++){
		structures[j] = (char* ) space(sizeof(char)*(len+1));
		structures[j] = pbacktrack(seq);
	}
	
    distMat = distMatrix(eN, structures);

    kmedoids(nclust, eN, distMat, npass, clustid, &error, &ifound);

    // Scoring
    // float **board[nMax - nMin][len - nMin];
	float **board = (float**)malloc((nMax - nMin) * sizeof(float*));
    for(i = 0; i < (nMax-nMin); i++) {
		board[i] = (float *)malloc((len - nMin) * sizeof(float));
		for (j=0; j < (len-nMin); j++){
			board[i][j] = 0;
		}
	}
	
	// memset(board, 0, (nMax - nMin) * (len - nMin) * sizeof(float));
    // Using BP
    if(globalArgs.probability == 1) {
	printf("Using Probability \n");
	scoreBP(nMin, nMax, len, board, matrix, sp, sn);
    }
    // Using Sampled Ensemble
    else {
	printf("Using Ensemble \n");
	scoreEnsemble(nMin, nMax, len, eN, board, structures, sp, sn);
    }

    // Find Max
    int winI = 0;
    int kI = 0;	

    findMax(&winI, &kI, nMax - nMin, len - nMin, board);
    // model_detailsT md;
    // paramT *parameters = get_scaled_parameters(temperature, md);

    winI = winI + nMin;
    // Generating constraints
    for(int k = 0; k < kI; k++) {
	foldStructUP[k] = '.';
	foldStructDOWN[k] = '.';
    }
    for(int k = kI; k <= kI + winI; k++) {
	foldStructUP[k] = '<';
	foldStructDOWN[k] = '>';
    }
    for(int k = kI + winI + 1; k < len; k++) {
	foldStructUP[k] = '.';
	foldStructDOWN[k] = '.';
    }
    // Folding the two structures
    eUP = fold_par(seq, foldStructUP, NULL, 1, 0);
    eDOWN = fold_par(seq, foldStructDOWN, NULL, 1, 0);
    /*	int pi,pj;
            for (pi=0;pi<nMax-nMin;pi++){
                    for(pj=0;pj<len-nMin;pj++){
                            printf("%f \t",board[pi][pj]);
                    }
                    printf("\n");
            }*/

    /*printf("Clustering results: \n");
    for(int j=0;j<eN;j++){
            printf("%d ",clustid[j]);
    }*/
    if(globalArgs.logging == 1) {
	printf("VERBOSE:\n\tSolution found %d times; within-cluster sum of distances is %f\n", ifound, error);
    }

    // Here goes silhuoette evaluation
    float silScore = 0;
    silhuoette_score2clust(clustid, distMat, eN, &silScore);

    // Indices:
    float iT, iSD, iTot;
    terminatorEvaluator(seq, foldStructUP, foldStructDOWN, len, &iT);
    SDEvaluator(seq, foldStructUP, foldStructDOWN, len, &iSD);
    iTot = (1 - iT) * iSD + iT + silScore;

    // Writing to File
    if(globalArgs.wFile == 1) {
	char* filePath = globalArgs.filePath;
	FILE* fp;
	fp = fopen(filePath, "w");
	if(!fp) {
	    printf("ERROR WRITING FILE. \n");
	} else {
	    fprintf(fp, "Sequence: \n%s \nStructure Up: \n%s \nStructure Down: \n%s \nSwitching Seq: \n[%d,%d]\n", seq,
	        foldStructUP, foldStructDOWN, kI, winI);
	    fprintf(fp, "iT: %f\tiSD: %f\tiSil: %f\tiTot: %f\n", iT, iSD, silScore, iTot);
	    fclose(fp);
	    printf("File Written! \n");
	}
	free(filePath);
	free(fp);
    }
	
	
    /*
     * Writing results to cmdline
     *
     */
    gettimeofday(&tval_after, NULL);
    timersub(&tval_after, &tval_before, &tval_result);
    printf("Time elapsed: %ld.%06ld\n", (long int)tval_result.tv_sec, (long int)tval_result.tv_usec);
    printf("\nRESULTS:\nWinLen = %d, WinPos = %d \n", winI, kI);
    printf("Structures: \n");
    printf("%s \n", foldStructUP);
    printf("%s \n", foldStructDOWN);
    printf("iT: %f\tiSD: %f\tiSil: %f\tiTot: %f\n", iT, iSD, silScore, iTot);

    free_pf_arrays();
	free(matrix);
	free(board);
	free(distMat);
	free(structure);
	free(foldStructUP);
	free(foldStructDOWN);
	free(structures);
    return 0;
}

/*
 *
 * FUNCTIONS
 *
 *
 */

void scoreBP(int nMin, int nMax, int N, float** board, double **matrix, int sp, int sn)
{
    /* nMin		minumum windows length
     * nMax		maximum windows length
     * N		sequence length
     * board 	matrix
     * matrix	bp probability matrix
     *
     */
    // Internal cycling
    for(int winL = nMin; winL < nMax; winL++) {
	for(int k = 0; k < N - winL; k++) {
	    float pUp = 0, pDown = 0;

	    // Evaluate summations
	    for(int j = k; j < k + winL; j++) {
		for(int i = 0; i < j; i++) {
		    pUp += matrix[i][j];
		}
	    }
	    for(int j = k; j < k + winL; j++) {
		for(int i = j + 1; i < N; i++) {
		    pDown += matrix[i][j];
		}
	    }
	    float intpUp = sp * pUp + sn * pDown;
	    float intpDown = sp * pDown + sn * pUp;
	    board[winL - nMin][k] = intpUp * intpDown / pow(winL, 1.95);
	}
    }
}

// Function for find maximum index in an array
void findMax(int* winI, int* kI, int M, int N, float **board)
{
    float maximum = 0;
    for(int i = 0; i < M; i++) {
	for(int j = 0; j < N; j++) {
	    if(board[i][j] > maximum) {
		maximum = board[i][j];
		*winI = i;
		*kI = j;
	    }
	}
    }
}

// Scoring using ensemble of structures
void scoreEnsemble(int nMin,
    int nMax,
    int N,
    int eN,
    float **board,
    char** structures,
    int sp,
    int sn)
{
    /* nMin			minumum windows length
     * nMax			maximum windows length
     * N			sequence length
     * eN			ensemble size
     * board 		matrix
     * structures	pointer to the array of structures
     *
     */
    for(int winL = nMin; winL < nMax; winL++) {
	for(int k = 0; k < N - winL; k++) {
	    float pUp = 0, pDown = 0;
	    // Slide on Window
	    for(int lslide = k; lslide < k + winL; lslide++) {
		float vUp = 0, vDown = 0;
		for(int vslide = 0; vslide < eN; vslide++) {
		    if(structures[vslide][lslide] == '(') {
			vDown += sp;
			vUp += sn;
		    } else if(structures[vslide][lslide] == ')') {
			vDown += sn;
			vUp += sp;
		    }
		}
		pUp += vUp / ((float)eN);
		pDown += vDown / ((float)eN);
	    }
	    board[winL - nMin][k] = fabsf(pUp * pDown / (float)pow(winL, 1.95));
	}
    }
}

// Implementing construction of distance matrix, bp_distance is the only ammitted by now
double** distMatrix(int N, char** structures)
{
    double** distMat = (double**)malloc(sizeof(double*) * N);
	if (distMat == NULL){
		printf('\nError in initializing memory!\n');
		exit(EXIT_FAILURE);
		}
    for(int i = 0; i < N; i++) {
	distMat[i] = (double*)malloc(sizeof(double) * N);
	if (distMat[i] == NULL){
		printf('\nError in initializing memory!\n');
		exit(EXIT_FAILURE);
		}
	for(int j = 0; j < N; j++) {
	    distMat[i][j] = (double)bp_distance(structures[i], structures[j]);
	}
    }
    return distMat;
}

/*
 *
 * Functions measuring termination termination
 *
 */
void terminatorEvaluator(char* sequence, char* structure1, char* structure2, int N, float* iT)
{

    float iT1, iT2;
    termIndex(sequence, structure1, N, &iT1);
    termIndex(sequence, structure2, N, &iT2);
    *iT = fabsf(iT1 - iT2);
}

void termIndex(char* sequence, char* structure, int N, float* iT)
{
    char* foldT = (char*)space(sizeof(char) * N);
    strcpy(foldT, sequence);
    for(int i = 0; i < N; i++) {
	if(structure[i] == ')')
	    foldT[i] = 'x';
	else if(structure[i] == '(')
	    foldT[i] = 'y';
	else if(sequence[i] == 'U')
	    foldT[i] = 'T';
    }
    regex_t regexCompiled1;
    regex_t regexCompiled2;
    regcomp(&regexCompiled1, regexTerm1, REG_EXTENDED);
    regcomp(&regexCompiled2, regexTerm2, REG_EXTENDED);

    // Regexing
    if(!regexec(&regexCompiled1, foldT, (size_t)0, NULL, 0))
	*iT = 1;
    else if(!regexec(&regexCompiled2, foldT, (size_t)0, NULL, 0))
	*iT = 1;
    else
	*iT = 0;
    regfree(&regexCompiled1);
    regfree(&regexCompiled2);
}

/*
 * Evaluation of SD sequestering/releasing
 *
 *
 */
void SDEvaluator(char* sequence, char* structure1, char* structure2, int N, float* iSD)
{

    float iSD1, iSD2;
    SDIndex(sequence, structure1, N, &iSD1);
    SDIndex(sequence, structure2, N, &iSD2);
    *iSD = fabsf(iSD1 - iSD2);
}

void SDIndex(char* sequence, char* structure, int N, float* iSD)
{
    char* foldT = (char*)space(sizeof(char) * N);
    strcpy(foldT, sequence);
    for(int i = 0; i < N; i++) {
	if(structure[i] == ')' || structure[i] == '(')
	    foldT[i] = 'x';
    }
    size_t maxGroups = 1;
    size_t maxMatches = 1;
    regmatch_t groupArray[maxGroups];
    regex_t regexCompiled1;
    regex_t regexCompiled2;
    regex_t regexCompiled3;
    regex_t regexCompiled4;
    regex_t regexCompiled5;
    regex_t regexCompiled6;
    regex_t regexCompiled7;
    regex_t regexCompiled8;
    regex_t regexCompiled9;
    regex_t regexCompiled10;

    regcomp(&regexCompiled1, regexSD1, REG_EXTENDED);
    regcomp(&regexCompiled2, regexSD2, REG_EXTENDED);
    regcomp(&regexCompiled3, regexSD3, REG_EXTENDED);
    regcomp(&regexCompiled4, regexSD4, REG_EXTENDED);
    regcomp(&regexCompiled5, regexSD5, REG_EXTENDED);
    regcomp(&regexCompiled6, regexSD6, REG_EXTENDED);
    regcomp(&regexCompiled7, regexSD7, REG_EXTENDED);
    regcomp(&regexCompiled8, regexSD8, REG_EXTENDED);
    regcomp(&regexCompiled9, regexSD9, REG_EXTENDED);
    regcomp(&regexCompiled10, regexSD10, REG_EXTENDED);

    // Regexing
    if(!regexec(&regexCompiled1, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else if(!regexec(&regexCompiled2, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else if(!regexec(&regexCompiled3, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else if(!regexec(&regexCompiled4, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else if(!regexec(&regexCompiled5, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else if(!regexec(&regexCompiled6, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else if(!regexec(&regexCompiled7, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else if(!regexec(&regexCompiled8, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else if(!regexec(&regexCompiled9, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else if(!regexec(&regexCompiled10, sequence, maxMatches, groupArray, 0)) {
	int start, finish;
	start = groupArray[0].rm_so;
	finish = start + sdLen;
	int count = 0;
	for(int j = start; j < finish; j++) {
	    if(foldT[j] == 'x')
		count++;
	}
	*iSD = count / ((float)sdLen);
    } else
	*iSD = 0;
    regfree(&regexCompiled1);
    regfree(&regexCompiled2);
    regfree(&regexCompiled3);
    regfree(&regexCompiled4);
    regfree(&regexCompiled5);
    regfree(&regexCompiled6);
    regfree(&regexCompiled7);
    regfree(&regexCompiled8);
    regfree(&regexCompiled9);
    regfree(&regexCompiled10);
}

/*
 * Silhouette score computation
 *
 *
 */

void silhuoette_score2clust(int* clustid, double** distMat, int N, float* silScore)
{
    /*
     * Only works with two clusters!
     *
     */
    float ai, bi, countIn, countOut;
    float iValued = 0;
    for(int i = 0; i < N; i++) {
	int cID = clustid[i];
	countIn = 0;
	countOut = 0;
	ai = 0;
	bi = 0;
	for(int j = 0; j < N; j++) {
	    if(cID == clustid[j]) {
		countIn++;
		ai += distMat[i][j];
	    } else if(cID != clustid[j]) {
		countOut++;
		bi += distMat[i][j];
	    }
	}
	ai = ai / (countIn - 1);
	bi = bi / countOut;
	iValued += (bi - ai) / max(ai, bi);
    }
    *silScore = iValued / N;
}

/*
 * Verify Sequence
 *
 */

int verifySequence(char* sequence, int len)
{
    for(int k = 0; k < len; k++) {
	sequence[k] = toupper(sequence[k]);
	if(sequence[k] == 'U') {
	    sequence[k] = 'T';
	}
	if(sequence[k] != 'A' && sequence[k] != 'C' && sequence[k] != 'G' && sequence[k] != 'T' && sequence[k] != 'U') {
	    printf("\n\n%c is not an admissible characther. [Must be in 'A','C','G',T','U']\n\n", sequence[k]);
	    return 1;
	}
    }
    return 0;
}